/*
 * Implement a "channel" abstraction
 *
 * This is just an array of file descriptors which will be polled
 * for activity.  Each file descriptor is associated with a "channel". 
 * Each channel also has buffers associated to hold data for pending read
 * or write operations, and completion routines which are called when the
 * I/O is complete.
 */
#include <stdio.h>
#include <stdlib.h>
#ifndef _WIN32
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#else
#include "lf_win.h"
#endif
#include <sys/select.h>

#include "libfma.h"
#include "lf_internal.h"
#include "lf_channel.h"

/*
 * static prototypes
 */
static int handle_rx_ready(struct lf_channel *chp);
static int handle_tx_ready(struct lf_channel *chp);
static int handle_hangup(struct lf_channel *chp);

/*
 * Local storage
 */

/* channel management info */
struct lf_channel_desc {
  struct pollfd *poll_array;		/* structs for poll() */
  struct lf_channel **channel_array;	/* links to associated channels */
  int poll_array_size;		/* number of poll entries alloced */
  int poll_count;			/* current number of poll items */
  int array_changed;
};
static struct lf_channel_desc *C;

/*
 * Initialize channel structures - this uses local static storage
 */
int
lf_init_channels()
{
  /* allocate the structure for keeping I/O variables */
  LF_CALLOC(C, struct lf_channel_desc, 1);

  /* Set up the poll array */
  C->poll_array = NULL;
  C->poll_array_size = 0;
  C->channel_array = NULL;	/* same "size" as poll array */
  C->poll_count = 0;

  return 0;

 except:
   LF_FREE(C);
   return -1;
}

/*
 * Create an association for a connection.  It is 
 * the responsibility of the owner to fill in completion routines and 
 * buffer pointers.
 */
int
lf_add_channel(
  struct lf_channel *chp)
{
  int i;

  /* Increment number of poll entries, saving old value */
  i = C->poll_count++;

  /* allocate more if necessary */
  if (C->poll_count > C->poll_array_size) {
    void *p;

    C->poll_array_size += FMS_IO_SIZE_INCR;

    p = (struct pollfd *) realloc(C->poll_array,
    				C->poll_array_size * sizeof(struct pollfd));
    if (p == NULL) {
      LF_ERROR(("growing poll_array"));
    } else {
      C->poll_array = (struct pollfd *)p;
    }

    p = (struct lf_channel **) realloc(C->channel_array,
		      C->poll_array_size * sizeof(struct channel *));
    if (p == NULL) {
      LF_ERROR(("growing channel_array"));
    } else {
      C->channel_array = (struct lf_channel **)p;
    }
  }

  /* Fill in the pollfd struct */
  C->poll_array[i].fd = chp->fd;
  C->poll_array[i].events = POLLHUP;

  /* Save pointer to channel structure */
  C->channel_array[i] = chp;
  chp->poll_index = i;	/* save reverse pointer */

  C->array_changed = TRUE;	/* flag that the array changed */

  return 0;

 except:
  return -1;
}

/*
 * Destroy an I/O association - take this connection out of the list to
 * be polled.
 */
void
lf_remove_channel(
  struct lf_channel *chp)
{
  int i;

  /* decrement number of items to be polled */
  --C->poll_count;

  i = chp->poll_index;	/* slot we are vacating */

  /* If we just trimmed the last one, nothing else to do */
  if (i == C->poll_count) {
    return;
  }

  /* otherwise, we need to copy the last one into the slot we just vacated */
  C->poll_array[i] = C->poll_array[C->poll_count];
  C->channel_array[i] = C->channel_array[C->poll_count];

  /* and fix up the back pointer */
  C->channel_array[i]->poll_index = i;

  C->array_changed = TRUE;	/* flag that the array changed */
}

/*
 * process any channels with data ready to send
 */
static int
handle_tx_ready(
  struct lf_channel *chp)
{
  int nw;

  /* send as much data as we can */
  if (chp->tx_count > 0) {
    nw = send(chp->fd, chp->tx_buffer, chp->tx_count, 0);
    if (nw == -1) LF_ERROR(("writing data"));

    /* adjust pointer and count */
    chp->tx_buffer = (char*)chp->tx_buffer + nw;
    chp->tx_count -= nw;
  }

  /* if nothing left to write, call the service routine */
  if (chp->tx_count == 0) {
    chp->tx_rtn(chp);
  }
  return 0;

 except:
  return -1;
}

/*
 * process any channels with data ready to read
 */
static int
handle_rx_ready(
  struct lf_channel *chp)
{
  int nr;

  /* read as much data as we can */
  if (chp->rx_count > 0) {
    nr = recv(chp->fd, chp->rx_buffer, chp->rx_count, 0);

    /* treat an error or 0-byte read at this point as a hangup */
    if (nr <= 0) {
      int rc;

      rc = handle_hangup(chp);
      if (rc != 0) LF_ERROR(("Handling implicit hangup"));
      return 0;
    }

    /* adjust pointer and count */
    chp->rx_buffer = (char*)chp->rx_buffer + nr;
    chp->rx_count -= nr;
  }

  /* if nothing left to read, call the service routine */
  if (chp->rx_count == 0) {
    chp->rx_rtn(chp);
  }
  return 0;

 except:
  return -1;
}

/*
 * handle a channel which hung up (disconnected)
 */
static int
handle_hangup(
  struct lf_channel *chp)
{
  if (chp->hangup_rtn != NULL) {
    chp->hangup_rtn(chp);
  }
  return 0;
}

/*
 * poll for and handle I/O
 * This is part of the main loop of the Fabric Manager
 */
int
lf_poll_channels(
  int timeout)
{
  int np;
  int i;
  int rc;

#if FMS_USE_SELECT
  int fd;
  fd_set rfd;
  fd_set wfd;
  fd_set efd;
  struct timeval *top;
  struct timeval to;
  int nfd;

 restart:
  C->array_changed = FALSE;	/* clear changed flag */
  FD_ZERO(&wfd);
  FD_ZERO(&rfd);
  FD_ZERO(&efd);
  nfd = 0;

  for (i=0; i<C->poll_count; ++i) {
    fd = C->poll_array[i].fd;
    if (C->poll_array[i].events & POLLIN) {
      FD_SET(fd, &rfd);
    }
    if (C->poll_array[i].events & POLLOUT) {
      FD_SET(fd, &wfd);
    }
    if (C->poll_array[i].events & (POLLERR|POLLHUP)) {
      FD_SET(fd, &efd);
    }

    if (C->poll_array[i].fd >= nfd) {
      nfd = C->poll_array[i].fd + 1;
    }
  }

  if (timeout >= 0) {
    to.tv_sec = timeout / 1000;
    to.tv_usec = (timeout % 1000) * 1000;
    top = &to;
  } else {
    top = NULL;
  }


  np = select(nfd, &rfd, &wfd, &efd, top);
  if (np == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
      && errno != ERESTART
#endif
      && errno != EAGAIN) {
    LF_ERROR(("Error from select()"));
  }

  /* loop over each file desc to see if it needs attention */
  if (np > 0) {
    for (i = 0; i < C->poll_count; ++i) {

      fd = C->poll_array[i].fd;

      /* ready for read ? */
      if (FD_ISSET(fd, &rfd)) {
	rc = handle_rx_ready(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling receive"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }

      /* ready for write ? */
      if (FD_ISSET(fd, &wfd)) {
	rc = handle_tx_ready(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling transmit"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }

      /* hangup? */
      if (FD_ISSET(fd, &efd)) {
	rc = handle_hangup(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling hangup"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }
    }
  }
  

#else	/* ! FMS_USE_SELECT */
  struct pollfd *pp;

 restart:
  C->array_changed = FALSE;	/* clear changed flag */

  /* execute the poll */
#if 0
printf("poll n=%d, timeout=%d              \n", C->poll_count, timeout);
fflush(stdout);
#endif
  np = poll(C->poll_array, C->poll_count, timeout);
  if (np == -1 && errno != EINTR
#if HAVE_DECL_ERESTART
      && errno != ERESTART
#endif
      && errno != EAGAIN) {
    LF_ERROR(("poll"));
  }

  /* loop over each file desc to see if it needs attention */
  if (np > 0) {
    for (i = 0, pp = C->poll_array;
	 i < C->poll_count;
	 ++i, ++pp) {

      /* POLLNVAL is a hard error */
      if (pp->revents & POLLNVAL) {
        LF_ERROR(("poll returned POLLNVAL"));
      }

      /* hangup? */
      if (pp->revents & (POLLHUP|POLLERR)) {
	rc = handle_hangup(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling hangup"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }

      /* ready for read ? */
      if (pp->revents & (POLLIN|POLLPRI)) {
	rc = handle_rx_ready(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling receive"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }

      /* ready for write ? */
      if (pp->revents & POLLOUT) {
	rc = handle_tx_ready(C->channel_array[i]);
	if (rc == -1) LF_ERROR(("handling transmit"));

	/* restart if the list has changed */
	if (C->array_changed) {
	  timeout = 0;
	  goto restart;
	}
      }
    }
  }
#endif /* FMS_USE_SELECT */

  return 0;

 except:
  return -1;
}

/*
 * Set up for receiving data on a channel
 */
void
lf_channel_receive(
  struct lf_channel *chp,
  void *buf,
  uint32_t len,
  lf_channel_handler_t rxrtn)
{
  chp->rx_rtn = rxrtn;
  chp->rx_buffer = buf;
  chp->rx_count = len;

  /* poll this channel for incoming data */
  C->poll_array[chp->poll_index].events |= POLLIN;
}

/*
 * Consume and discard all data until the socket closes
 */
void
lf_channel_data_drain(
  struct lf_channel *chp)
{
  static char dumpbuf[1024];

  lf_channel_receive(chp, dumpbuf, sizeof(dumpbuf), lf_channel_data_drain);
}
